<#
.SYNOPSIS
    Removes selected items from the CCMCache.

.DESCRIPTION
    Removes selected items from the CCMCache.

.EXAMPLE
     Start the application and show the GUI.
     ."ClearSoftwareCenterCache_v01.ps1" 

.EXAMPLE
     Start the application, show the GUI in the Italian GUI language.
     ."ClearSoftwareCenterCache_v01.ps1" -LanguageOverride it
     The following languages are available:
       - en -> English
       - de -> German
       - fr -> French
       - nl -> Dutch
       - it -> Italian
       - sp -> Spanish
       - id -> Indonesian

.NOTES
    Author:  Willem-Jan Vroom
    Website: 
    Twitter: @TheStingPilot

v0.1 Beta 1:
   * Initial version. 

v0.1 Beta 2:
   * Corrected the translation of the sentence 'You need elevated (admin) rights to run this script.'
   * Modified the function CCM-FindOrphanedCCMFolders.


Resources:
 - https://SCCM.Zone/Clean-CMClientCache

#>

[CmdletBinding()]

Param
  (
   [Parameter(HelpMessage='Enable detailed logging to a log file.')]
   [Switch]   $DetailedLogging,

   [Parameter(HelpMessage = 'Override the language.')]
   [ValidateSet('en','de',"nl","fr","es","it","id")]
   [String]   $LanguageOverride
  )

# =============================================================================================================================================
# Function block
# =============================================================================================================================================

  Function Display-MessageBox
   {

    <#
    .NOTES
    =============================================================================================================================================
    Created with:     Windows PowerShell ISE
    Created on:       24-May-2021 / 24-Apr-2024 / 09-June-2022
    Created by:       Willem-Jan Vroom
    Organization:     
    Functionname:     Display-MessageBox
    =============================================================================================================================================
    .SYNOPSIS

    This function displays a message box.

    The return value is:

    None (no result) 	0
    OK 	                1
    Cancel 	            2
    Yes 	            6
    No. 	            7

    #>

    param
     (
      [Parameter(Mandatory=$True)] [ValidateNotNullOrEmpty()][String] $Text,
      [Parameter(Mandatory=$False)][Switch] $GUI,
      [Parameter(Mandatory=$False)][Switch] $Error,
      [Parameter(Mandatory=$False)][ValidateSet('Ok','AbortRetryIgnore','OkCancel','YesNoCancel','YesNo')][String] $Button = "Ok"
     )
         
    If ($GUI)
     {
      If ($Error)
       {
        $Return = [System.Windows.MessageBox]::Show($Text.Trim(),$($CCMCacheActions.$Language.Error),$Button,"Error")
       }
        else
       {
        $Return = [System.Windows.MessageBox]::Show($Text.Trim(),$($CCMCacheActions.$Language.Information),$Button,"Asterisk")
       }
     }
      else
     {
      Write-Host "$Text`n"
      Return 0
     }

     Return $($Return.value__)
   }

  Function Add-EntryToLogFile
   {

    <#
    .NOTES
    =============================================================================================================================================
    Created with:     Windows PowerShell ISE
    Created on:       17-May-2020 / Modified 09-May-2022: Includes the function name
    Created by:       Willem-Jan Vroom
    Organization:     
    Functionname:     Add-EntryToLogFile
    =============================================================================================================================================
    .SYNOPSIS

    This function adds a line to a log file

    #>

    Param
     (
      [Parameter(Mandatory=$True)]  [string] $Entry,
      [Parameter(Mandatory=$False)] [String] $FunctionName

     )
      
     Write-Verbose "[Function: $FunctionName] - $Entry"
     If ($Global:gblDetailedLogging -and $Global:gblLogFile)
      {
       $Timestamp = (Get-Date -UFormat "%a %e %b %Y %X").ToString()
       Add-Content $Global:gblLogFile -Value $($Timestamp + "[Function: $FunctionName] - $Entry") -Force -ErrorAction SilentlyContinue
      }
   }

  Function Add-EntryToResultsFile
   {

    <#
    .NOTES
    =============================================================================================================================================
    Created with:     Windows PowerShell ISE
    Created on:       03-August-2018
    Created by:       Willem-Jan Vroom
    Organization:     
    Functionname:     Add-EntryToResultsFile
    =============================================================================================================================================
    .SYNOPSIS

    This function adds the success or failure information to the array that contains the log
    information.

    #>

    Param
     (
      [String]   $Name                  = "",
      [String]   $DeploymentType        = "",
      [String]   $Program               = "",
      [String]   $Evaluation            = "",
      [String]   $UpdateClassification  = "",
      [String]   $ContentID             = "",
      [String]   $ContentLocation       = "",
      [String]   $ContentSize           = "",
      [String]   $LastReferenceTime     = ""
     )

    $ThisFunctionName = $MyInvocation.InvocationName

    $Record  = [ordered] @{"Name"                        = "";
                           "Deployment Type"             = "";
                           "Program"                     = "";
                           "Evaluation"                  = "";
                           "Update Classification"       = "";
                           "Content ID"                  = "";
                           "Content Location"            = "";
                           "Content Size"                = "";
                           "Content Last Reference Time" = ""}

    $Record."Name"                        = $Name
    $Record."Deployment Type"             = $DeploymentType
    $Record."Program"                     = $Program
    $Record."Evaluation"                  = $Evaluation
    $Record."Update Classification"       = $UpdateClassification
    $Record."Content ID"                  = $ContentID
    $Record."Content Location"            = $ContentLocation
    $Record."Content Size"                = $ContentSize
    $Record."Content Last Reference Time" = $LastReferenceTime

    $Global:arrTable                     += New-Object PSObject -Property $Record
    
    Add-EntryToLogFile -FunctionName $ThisFunctionName -Entry "Discovered application, program or update:"
    Add-EntryToLogFile -FunctionName $ThisFunctionName -Entry "  * Name:                        $Name"
    Add-EntryToLogFile -FunctionName $ThisFunctionName -Entry "  * Deployment Type:             $DeploymentType"
    Add-EntryToLogFile -FunctionName $ThisFunctionName -Entry "  * Program:                     $Program"
    Add-EntryToLogFile -FunctionName $ThisFunctionName -Entry "  * Evaluation:                  $Evaluation"
    Add-EntryToLogFile -FunctionName $ThisFunctionName -Entry "  * Update Classification:       $UpdateClassification"
    Add-EntryToLogFile -FunctionName $ThisFunctionName -Entry "  * Content ID:                  $ContentID"
    Add-EntryToLogFile -FunctionName $ThisFunctionName -Entry "  * Content Location:            $ContentLocation"
    Add-EntryToLogFile -FunctionName $ThisFunctionName -Entry "  * Content Size:                $ContentSize"
    Add-EntryToLogFile -FunctionName $ThisFunctionName -Entry "  * Content Last Reference Time: $LastReferenceTime"
    Add-EntryToLogFile -FunctionName $ThisFunctionName -Entry "#######################################################################"
   }


  Function Create-Folder
   {

    <#
    .NOTES
    =============================================================================================================================================
    Created with:     Windows PowerShell ISE
    Created on:       03-August-2018
    Created by:       Willem-Jan Vroom
    Organization:     
    Functionname:     Create-Folder
    =============================================================================================================================================
    .SYNOPSIS

    This function creates the given folder.
    The function returns two results:
     1. A message.
     2. True or false (success or failure)

    #>
    
    param
     (
      [Parameter(Mandatory=$True)] [ValidateNotNullOrEmpty()][String] $FolderName
     )

   $bolResult     = $True
   $ResultMessage = ""

   If (-not(Test-Path $('FileSystem::' + $FolderName)))
    {
     New-Item -Path $FolderName -ItemType Directory | Out-Null
     Sleep 1
     If (test-path $('FileSystem::' + $FolderName))
      {
       $ResultMessage = $($CCMCacheActions.$Language.FolderCreated).Replace("<FolderName>", $FolderName)
      }
       else
      {
       $ResultMessage = $($CCMCacheActions.$Language.ErrorCreatingFolder).Replace("<FolderName>", $FolderName)
       $bolResult     = $False
      }
    }
     else
    {
     $ResultMessage = $($CCMCacheActions.$Language.FolderAlreadyExists).Replace("<FolderName>", $FolderName)
    }

    Return $ResultMessage,$bolResult 

   }

  Function Check-HasElevatedRights
   {

    <#
    .NOTES
    ========================================================================================================================
    Created with:     Windows PowerShell ISE
    Created on:       11-January-2019 / Modified 22-Apr-2022 / Modified 05-May-2022
    Created by:       Willem-Jan Vroom
    Organization:     
    Functionname:     Check-HasElevatedRights
    ========================================================================================================================
    .SYNOPSIS

    This function checks if an user has admin rights. The function returns $True or $False

    Initially, the function was ([Security.Principal.WindowsPrincipal] [Security.Principal.WindowsIdentity]::GetCurrent()).IsInRole([Security.Principal.WindowsBuiltInRole]'544')
    But that command throws up an error due to the PowerShell ConstrainedMode.
    "S-1-5-32-544" -in (whoami /groups /FO csv | convertfrom-csv).sid does not work either as it will always return 'true', even if not elevated but
    the logged on user is an admin
    So the only good test: check if you can create a file in c:\windows\fonts. If yes, then admin / elevated rights otherwise not. 

    #>

    $RandomFile = ""
    For($a=1; $a -le 8; $a++)
     {
      $RandomFile += [char](Random(97..122))
     }

    $RandomFile = "$($env:windir)\Fonts\$RandomFile.txt"

    Try
     {
      New-Item $RandomFile -Force -ErrorAction SilentlyContinue | Out-Null
      If (test-path($RandomFile))
       {
        Remove-Item $RandomFile -Force
        Return $True
       }
        else
       {
        Return $False
       }
     }
      Catch
     {  
      Return $False
     } 
   }

  Function Test-RegistryKeyValue
   {
    <#
    .NOTES
    =============================================================================================================================================
    Created with:     Windows PowerShell ISE
    Created on:       30-Dec-20
    Created by:       (C) Aaron Jensen 
                      https://stackoverflow.com/questions/5648931/test-if-registry-value-exists
    Organization:     Carbon Module
    Functionname:     Test-RegistryKeyValue
    =============================================================================================================================================
    .SYNOPSIS

    #>

    [CmdletBinding()]
    param 
     (
      [Parameter(Mandatory = $True)][string]  $Path,
      [Parameter(Mandatory = $True)][string]  $Name
     )

    if ( -not(Test-Path -Path $Path -PathType Container))
     {
      return $False
     }

    $properties = Get-ItemProperty -Path $Path
    if (-not $properties)
     {
      return $False
     } 

    $member = Get-Member -InputObject $properties -Name $Name
    if ($member)
     {
      return $True
     }
      else
     {
      return $False
     }
   }

  Function UserDetails
   {
    <#
    .NOTES
    =============================================================================================================================================
    Created with:     Windows PowerShell ISE
    Created on:       03-Jan-21 / Modified on 14-Jan-21 / Modified on 22-Apr-22 / Modified on 01-May-2022 / Modified on 17-May-2022
    Created by:       Willem-Jan Vroom
    Functionname:     UserDetails
    =============================================================================================================================================
    .SYNOPSIS

    This function returns 4 details of the Current Logged In Usser
    1. The username of the current logged in user
    2. User\Domain of the current logged in user
    3. User SID fo the User\Domain
    4. Account name that is using the script 

    Initially, the scriptAccount was found with the command 
    [System.Security.Principal.WindowsIdentity]::GetCurrent().Name

    But that command throws up an error due to the PowerShell ConstrainedMode.

    Also, the output of whoami /user depends on the language. In the German language the output is different.
    The header information is in the German language, so the header should be read from the CSV file.
    That can be done with PSObject.Properties.Name

    C:\Users\Test>whoami /user

    BENUTZERINFORMATIONEN
     ---------------------

    Benutzername         SID
    ==================== ==============================================
    windows10wkg_02\test S-1-5-21-1753037473-1212035439-1379364385-1001

    #>

    $Explorer      = (Get-WMIObject -Query "Select * From Win32_Process Where (Name='explorer.exe' or Name='pfwsmgr.exe')")
    If ($Explorer.Count -gt 1)
     {
      $UserName      = ($Explorer[-1]).GetOwner()
      $SID           = (($Explorer[-1]).GetOwnerSID()).SID
     }
      else
     {
      $UserName      = ($Explorer).GetOwner()
      $SID           = (($Explorer).GetOwnerSID()).SID
     }

    $UserAndDomain      = "$($Username.Domain )\$($Username.User)".ToUpper()
    $tmpScriptAccount   = (whoami /user /FO csv | convertfrom-csv)
    $TranslatedUserName = $tmpScriptAccount.PSObject.Properties.Name[0]
    $ScriptAccount      = $($tmpScriptAccount.$TranslatedUserName).ToUpper()
    
    Return $($Username.User),$UserAndDomain,$SID,$ScriptAccount

   }

  Function Find-Language
   {
    <#
    .NOTES
    =============================================================================================================================================
    Created with:     Windows PowerShell ISE
    Created on:       30-Dec-20
    Created by:       Willem-Jan Vroom 
    Organisation:                      
    Functionname:     Find-Language
    =============================================================================================================================================
    .SYNOPSIS

    This function works the best, even running under system context.
    Get-UiCulture returns the UICulture details from the account that is used. So when the system account is used, incorrect
    details might be shown.

    #>

    [CmdletBinding()]
    Param
     (
      [Parameter(Mandatory = $True)][ValidateNotNullOrEmpty()][String] $CurrentUserSID
     )

    $Result           = "en-US"
    $ThisFunctionName = $MyInvocation.InvocationName

    $RegKey = "REGISTRY::HKEY_USERS\$CurrentUserSID\Control Panel\Desktop"
    $Value  = "PreferredUILanguages"
    if (Test-RegistryKeyValue -Path $RegKey -Name $Value)
     {
      $Result = (get-itemproperty $RegKey | Select -ExpandProperty $Value).Split()[0]
      Add-EntryToLogFile -FunctionName $ThisFunctionName -Entry "Regkey '$RegKey' value '$Value' exists. The data is '$Result'."
      Return $Result
     }
    else
     {
      Add-EntryToLogFile -FunctionName $ThisFunctionName -Entry "Regkey '$RegKey' value '$Value' does not exists. Skipping to machine settings."
     }

    $RegKey = "REGISTRY::HKEY_USERS\.DEFAULT\Control Panel\Desktop\MuiCached"
    $Value  = "MachinePreferredUILanguages"
    if (Test-RegistryKeyValue -Path $RegKey -Name $Value)
     {
      $Result = (get-itemproperty $RegKey | Select -ExpandProperty $Value).Split()[0]
      Add-EntryToLogFile -FunctionName $ThisFunctionName -Entry "Regkey '$RegKey' value '$Value' exists. The data is '$Result'."
      Return $Result
     }

    Add-EntryToLogFile -FunctionName $ThisFunctionName -Entry "There was a problem reading the registry..."
    Return $Result
   }

  Function Check-SCCMClient
   {
    
    <#
    .NOTES
    =============================================================================================================================================
    Created with:     Windows PowerShell ISE
    Created on:       14-Jan-21
    Created by:       Willem-Jan Vroom 
    Organisation:                      
    Functionname:     Check-SCCMClient
    =============================================================================================================================================
    .SYNOPSIS

    This function checks if the SCCM client on the machine is operational. It returns either nothing or an error message.

    #>

    $ErrorMessage             = ""
    $ThisFunctionName         = $MyInvocation.InvocationName

    Add-EntryToLogFile -FunctionName $ThisFunctionName -Entry $CCMCacheActions.$Language.StartCheckSCCMClient

    Try
     {
      $class = Get-WmiObject -Class 'SMS_CLIENT' -List -Namespace 'root\ccm' -ErrorAction Stop
      Add-EntryToLogFile -FunctionName $ThisFunctionName -Entry $CCMCacheActions.$Language.SCCMClientOk
      Add-EntryToLogFile -FunctionName $ThisFunctionName -Entry $CCMCacheActions.$Language.EndCheckSCCMClient
      Return $True,$ErrorMessage
     }
      Catch
     {
      $ErrorMessage = $($_.Exception.Message).Trim()
      Add-EntryToLogFile -FunctionName $ThisFunctionName -Entry $CCMCacheActions.$Language.SCCMClientError1
      Add-EntryToLogFile -FunctionName $ThisFunctionName -Entry $($CCMCacheActions.$Language.SCCMClientError1).Replace("<ErrorMessage>",$ErrorMessage)
      Add-EntryToLogFile -FunctionName $ThisFunctionName -Entry $CCMCacheActions.$Language.EndCheckSCCMClient
      Return $False,$ErrorMessage
     }
   }

  Function CCM-FindApplications
   {

    <#
    .NOTES
    =============================================================================================================================================
    Created with:     Windows PowerShell ISE
    Created on:       06-June-2022
    Created by:       Willem-Jan Vroom 
    Organisation:                      
    Functionname:     CCM-FindApplications
    =============================================================================================================================================
    .SYNOPSIS

    This function finds all the via SCCM (installed) applications.

    The EnforcePreference is explained here:
    https://docs.microsoft.com/en-us/mem/configmgr/develop/reference/core/clients/sdk/ccm_application-client-wmi-class

    The EvaluationState is explained here:
    https://docs.microsoft.com/en-us/mem/configmgr/develop/reference/core/clients/sdk/ccm_evaluationstate-client-wmi-class

    #>

    $Applications             = Get-CimInstance -Namespace "Root\ccm\ClientSDK" -ClassName 'CCM_Application' -Verbose:$False
    $CCMComObject             = New-Object -ComObject 'UIResource.UIResourceMgr'
    $CacheInfo                = $($CCMComObject.GetCacheInfo().GetCacheElements())
    $ThisFunctionName         = $MyInvocation.InvocationName

    $EnforcePreference = @{
    0 = "Immediate"
    1 = "NonBusinessHours"
    2 = "AdminSchedule"}

    $EvaluationSate = @{
    0 = "No state information is available."
    1 = "Application is enforced to desired/resolved state."
    2 = "Application is not required on the client."
    3 = "Application is available for enforcement (install or uninstall based on resolved state). Content may/may not have been downloaded."
    4 = "Application last failed to enforce (install/uninstall)."
    5 = "Application is currently waiting for content download to complete."
    6 = "Application is currently waiting for content download to complete."
    7 = "Application is currently waiting for its dependencies to download."
    8 = "Application is currently waiting for a service (maintenance) window."
    9 = "Application is currently waiting for a previously pending reboot."
    10 = "Application is currently waiting for serialized enforcement."
    11 = "Application is currently enforcing dependencies."
    12 = "Application is currently enforcing."
    13 = "Application install/uninstall enforced and soft reboot is pending."
    14 = "Application installed/uninstalled and hard reboot is pending."
    15 = "Update is available but pending installation."
    16 = "Application failed to evaluate."
    17 = "Application is currently waiting for an active user session to enforce."
    18 = "Application is currently waiting for all users to logoff."
    19 = "Application is currently waiting for a user logon."
    20 = "Application in progress, waiting for retry."
    21 = "Application is waiting for presentation mode to be switched off."
    22 = "Application is pre-downloading content (downloading outside of install job)."
    23 = "Application is pre-downloading dependent content (downloading outside of install job)."
    24 = "Application download failed (downloading during install job)."
    25 = "Application pre-downloading failed (downloading outside of install job)."
    26 = "Download success (downloading during install job)."
    27 = "Post-enforce evaluation."
    28 = "Waiting for network connectivity."}

    ForEach ($Application in $Applications)
     {
      $tmpName                 = ""
      $tmpDeploymentType       = ""
      $tmpEvaluation           = ""
      $tmpContentID            = ""
      $tmpContentLocation      = ""
      $tmpContentSize          = ""
      $tmpLastReferenceTime    = ""

      $AppDTS = ($Application | Get-CimInstance).AppDTs
      ForEach ($AppDT in $AppDTS)
       {
        ForEach ($ActionType in $AppDT.AllowedActions)
         {
          $Arguments = [hashtable]@{
          'AppDeliveryTypeID' = [string]$($AppDT.ID)
          'Revision'          = [uint32]$($AppDT.Revision)
          'ActionType'        = [string]$($ActionType)}
          $AppContentID = (Invoke-CimMethod -Namespace 'Root\ccm\cimodels' -ClassName 'CCM_AppDeliveryType' -MethodName 'GetContentInfo' -Arguments $Arguments -Verbose:$False).ContentID

          $tmpName            = $Application.Name
          $tmpDeploymentType  = "$($AppDT.Name) -> Action: $ActionType"
          $tmpEvaluation      = $($EvaluationSate.Item([int]$($Application.EvaluationState)))
           

          Add-EntryToLogFile -FunctionName $ThisFunctionName -Entry "Application: Name:                 $tmpName"
          Add-EntryToLogFile -FunctionName $ThisFunctionName -Entry "Application: Enforce preference:   $($EnforcePreference.Item([int]$($Application.EnforcePreference)))"
          Add-EntryToLogFile -FunctionName $ThisFunctionName -Entry "Application: Evaluation:           $tmpEvaluation"
          Add-EntryToLogFile -FunctionName $ThisFunctionName -Entry "Application: Deployment Type name: $($AppDT.Name)"
          Add-EntryToLogFile -FunctionName $ThisFunctionName -Entry "Application: Action:               $($ActionType)"
 
          $AppCacheInfo = $CacheInfo | Where-Object { $($_.ContentID) -eq $AppContentID }
          If ($AppCacheInfo)
           {
            $tmpContentID         = $AppCacheInfo.ContentID
            $tmpContentLocation   = $AppCacheInfo.Location
            $tmpContentSize       = $AppCacheInfo.ContentSize
            $tmpLastReferenceTime = $AppCacheInfo.LastReferenceTime
         
            Add-EntryToLogFile -FunctionName $ThisFunctionName -Entry "Application: Content ID:           $tmpContentID"
            Add-EntryToLogFile -FunctionName $ThisFunctionName -Entry "Application: Location:             $tmpContentLocation"
            Add-EntryToLogFile -FunctionName $ThisFunctionName -Entry "Application: Content size:         $('{0:N2}' -f $($tmpContentSize / 1KB))Mb"
            Add-EntryToLogFile -FunctionName $ThisFunctionName -Entry "Application: Last Reference time:  $tmpLastReferenceTime"
           }
          Add-EntryToResultsFile -Name $tmpName -DeploymentType $tmpDeploymentType -Evaluation $tmpEvaluation -ContentID $tmpContentID -ContentLocation $tmpContentLocation -LastReferenceTime $tmpLastReferenceTime -ContentSize $tmpContentSize
         }
       }
     }
   }

  Function CCM-FindPrograms
   {

    <#
    .NOTES
    =============================================================================================================================================
    Created with:     Windows PowerShell ISE
    Created on:       06-June-2022
    Created by:       Willem-Jan Vroom 
    Organisation:                      
    Functionname:     CCM-FindPrograms
    =============================================================================================================================================
    .SYNOPSIS

    This function finds all the via SCCM (installed) programs.

    #>

    $Packages                 = Get-CimInstance -Namespace 'ROOT\ccm\Policy\Machine\RequestedConfig' -ClassName 'CCM_SoftwareDistribution' -Verbose:$False
    $CCMComObject             = New-Object -ComObject 'UIResource.UIResourceMgr'
    $CacheInfo                = $($CCMComObject.GetCacheInfo().GetCacheElements())
    $ThisFunctionName         = $MyInvocation.InvocationName

    ForEach ($Package in $Packages)
     {
      $tmpName                 = $Package.PKG_MIFName
      $tmpProgram              = $Package.PRG_ProgramID
      $tmpContentID            = ""
      $tmpContentLocation      = ""
      $tmpContentSize          = ""
      $tmpLastReferenceTime    = ""

      $PkgCacheInfo = $CacheInfo | Where-Object { $_.ContentID -eq $Package.PKG_PackageID }

      Add-EntryToLogFile -FunctionName $ThisFunctionName -Entry "Program: Name:                 $tmpName"
      Add-EntryToLogFile -FunctionName $ThisFunctionName -Entry "Program: Program name:         $tmpProgram"

      # Only take the last items from the cache information. 
      If (($($PkgCacheInfo | Measure-Object).Count) -gt 1){$PkgCacheInfo1 = $PkgCacheInfo[-1]} else {$PkgCacheInfo1 = $PkgCacheInfo}

      ForEach ($CacheItem in $PkgCacheInfo1)
       {
        If ($CacheItem) 
         {
          #  Set content size to 0 if null to avoid division by 0
          If ($CacheItem.ContentSize -eq 0) { [int]$PkgContentSize = 0 } Else { [int]$PkgContentSize = $($CacheItem.ContentSize) }
          $tmpContentID         = $CacheItem.ContentID
          $tmpContentLocation   = $CacheItem.Location
          $tmpContentSize       = $PkgContentSize
          $tmpLastReferenceTime = $CacheItem.LastReferenceTime

          Add-EntryToLogFile -FunctionName $ThisFunctionName -Entry "Package: Content ID:           $tmpContentID"
          Add-EntryToLogFile -FunctionName $ThisFunctionName -Entry "Package: Location:             $tmpContentLocation"
          Add-EntryToLogFile -FunctionName $ThisFunctionName -Entry "Package: Content size:         $('{0:N2}' -f $($PkgContentSize / 1KB))Mb"
          Add-EntryToLogFile -FunctionName $ThisFunctionName -Entry "Package: Last reference time:  $tmpLastReferenceTime"   
         }
       }
      Add-EntryToResultsFile -Name $tmpName -Program $tmpProgram -ContentID $tmpContentID -ContentLocation $tmpContentLocation -LastReferenceTime $tmpLastReferenceTime -ContentSize $tmpContentSize
     }
   }

  Function CCM-FindUpdates
   {

    <#
    .NOTES
    =============================================================================================================================================
    Created with:     Windows PowerShell ISE
    Created on:       06-June-2022
    Created by:       Willem-Jan Vroom 
    Organisation:                      
    Functionname:     CCM-FindUpdates
    =============================================================================================================================================
    .SYNOPSIS

    This function finds all the via SCCM installed updates.
    The Classification Type is explained here:
    https://docs.microsoft.com/en-us/previous-versions/windows/desktop/ff357803(v=vs.85)

    #>

    $CCMComObject             = New-Object -ComObject 'UIResource.UIResourceMgr'
    $CacheInfo                = $($CCMComObject.GetCacheInfo().GetCacheElements())
    $Updates                  = Get-CimInstance -Namespace 'Root\ccm\SoftwareUpdates\UpdatesStore' -ClassName 'CCM_UpdateStatus' -Verbose:$False
    $ThisFunctionName         = $MyInvocation.InvocationName

    $ClassificationType = @{
    "5C9376AB-8CE6-464A-B136-22113DD69801" = "Application"
    "434DE588-ED14-48F5-8EED-A15E09A991F6" = "Connectors"
    "E6CF1350-C01B-414D-A61F-263D14D133B4" = "CriticalUpdates"
    "E0789628-CE08-4437-BE74-2495B842F43B" = "DefinitionUpdates"
    "E140075D-8433-45C3-AD87-E72345B36078" = "DeveloperKits"
    "B54E7D24-7ADD-428F-8B75-90A396FA584F" = "FeaturePacks"
    "9511D615-35B2-47BB-927F-F73D8E9260BB" = "Guidance"
    "0FA1201D-4330-4FA8-8AE9-B877473B6441" = "SecurityUpdates"
    "68C5B0A3-D1A6-4553-AE49-01D3A7827828" = "ServicePacks"
    "B4832BD8-E735-4761-8DAF-37F882276DAB" = "Tools"
    "28BC880E-0592-4CBF-8F95-C79B17911D5F" = "UpdateRollups"
    "CD5FFD1E-E932-4E3A-BF74-18BF0B1BBD83" = "Updates"} 

    ForEach ($Update in $Updates)
     {
      $tmpName                 = "$($Update.Title) Bulletin ID: $($Update.Article)"
      $tmpEvaluation           = $Update.Status
      $tmpUpdateClassification = $($ClassificationType.Item($($Update.UpdateClassification)))
      $tmpContentID            = ""
      $tmpContentLocation      = ""
      $tmpContentSize          = ""
      $tmpLastReferenceTime    = ""

      Add-EntryToLogFile -FunctionName $ThisFunctionName -Entry "Update: Title and  Bulletin ID:   $tmpName"
      Add-EntryToLogFile -FunctionName $ThisFunctionName -Entry "Update: Evaluation:               $tmpEvaluation"

      $UpdateCacheInfo = $CacheInfo | Where-Object { $_.ContentID -eq $Update.UniqueID}
      If ($UpdateCacheInfo)
       {
        ForEach ($Update in $UpdateCacheInfo)
         {
          $tmpContentID            = $Update.ContentID
          $tmpContentLocation      = $Update.Location
          $tmpContentSize          = $Update.ContentSize
          $tmpLastReferenceTime    = $Update.LastReferenceTime

          Add-EntryToLogFile -FunctionName $ThisFunctionName -Entry "Update: Content ID:               $tmpContentID"
          Add-EntryToLogFile -FunctionName $ThisFunctionName -Entry "Update: Location:                 $tmpContentLocation"
          Add-EntryToLogFile -FunctionName $ThisFunctionName -Entry "Update: Content size:             $('{0:N2}' -f $($tmpContentSize / 1KB))Mb"
          Add-EntryToLogFile -FunctionName $ThisFunctionName -Entry "Update: Last reference time:      $tmpLastReferenceTime"
         }   

       }
      Add-EntryToResultsFile -Name $tmpName -Evaluation $tmpEvaluation -UpdateClassification $tmpUpdateClassification -ContentID $tmpContentID -ContentLocation $tmpContentLocation -LastReferenceTime $tmpLastReferenceTime  -ContentSize $tmpContentSize  
     }
   }

  Function CCM-FindOrphanedCCMFolders
   {

    <#
    .NOTES
    =============================================================================================================================================
    Created with:     Windows PowerShell ISE
    Created on:       06-June-2022
    Created by:       Willem-Jan Vroom 
    Organisation:                      
    Functionname:     CCM-FindOrphanedCCMFolders
    =============================================================================================================================================
    .SYNOPSIS

    This function finds all the folders in the CCMCache that are no longer needed or used. Also, there is no write action in that folder for
    at least one hour.

    Also, there might be folders that are still in the CCMCache, have details like ContentID but that content is no longer used by any application,
    program or update.

    Skip Content Locations that contains a dot. 
        
    #>

    $CCMComObject                    = New-Object -ComObject 'UIResource.UIResourceMgr'
    $CCMCacheFolderUsedByDeployments = $CCMComObject.GetCacheInfo().GetCacheElements()
    $CacheLocation                   = $CCMComObject.GetCacheInfo().Location
    $AllFoldersInCache               = (Get-ChildItem $CacheLocation -Directory).FullName
    $ThisFunctionName                = $MyInvocation.InvocationName
    
    $FoundOrhanedFolders             = @()

    $Record  = [ordered] @{"Orphaned Folder Name"  = "";
                           "ContentId"             = ""}

  # =============================================================================================================================================
  # Step 1: Check for folders in the CacheLocation (for example C:\windows\ccmcache) that are not used by any application, program or update.
  #         Add these folders to the array FoundOrhanedFolders
  # =============================================================================================================================================
  
    ForEach ($FolderInCache in $AllFoldersInCache)
     {
      If (($CCMCacheFolderUsedByDeployments | ForEach-Object {$_.Location}) -notcontains $FolderInCache)
       {
        if ((Get-ItemProperty $FolderInCache).LastWriteTime -le (get-date).AddHours(-1) -and -not $FolderInCache.Contains("."))
         {
          Add-EntryToLogFile -FunctionName $ThisFunctionName -Entry $($CCMCacheActions.$Language.OrphanedFolder).Replace("<FolderInCache>",$FolderInCache)
          $Record."Orphaned Folder Name" = $FolderInCache
          $objRecord                      = New-Object PSObject -Property $Record
          $FoundOrhanedFolders           += $objRecord
         }
       }
     }

  # =============================================================================================================================================
  # Step 2: The array Global:arrTable contains all the found applications, programs and updates, including the content location.
  #         Now, we check if there are folders in the CacheLocation (for example C:\windows\ccmcache) that are not in the array Global:arrTable. 
  #         Add these folders to the array FoundOrhanedFolders.
  # =============================================================================================================================================
  
    ForEach ($FolderInCache in ($CCMCacheFolderUsedByDeployments | ForEach-Object {$_.Location} | Where-Object {$_ -notin $Global:arrTable."Content Location"}))
     {
      $ContentID = $CCMCacheFolderUsedByDeployments | Where-Object {$_.Location -eq $FolderInCache} | ForEach-Object {$_.ContentId}
      $Record."Orphaned Folder Name" = $FolderInCache
      $Record."ContentId"            = $ContentID
      $objRecord                      = New-Object PSObject -Property $Record
      $FoundOrhanedFolders           += $objRecord
     }

    $FoundOrhanedFolders = $FoundOrhanedFolders | Sort-Object "Orphaned Folder Name"

    Return $FoundOrhanedFolders
   }

  Function CCM-DeleteItemFromCache
   {

    <#
    .NOTES
    =============================================================================================================================================
    Created with:     Windows PowerShell ISE
    Created on:       09-June-2022
    Created by:       Willem-Jan Vroom 
    Organisation:                      
    Functionname:     CCM-DeleteItemFromCache
    =============================================================================================================================================
    .SYNOPSIS

    This function removes the specified item from the cache.
        
    #>

    [CmdletBinding()]

    param
    (
     [Parameter(Mandatory = $True,  Position = 0)][ValidateNotNullOrEmpty()] [String]  $ContentID,
     [Parameter(Mandatory = $False, Position = 1)]                           [Boolean] $RemoveIfPersistentInCache = $False,
     [Parameter(Mandatory = $False, Position = 2)][ValidateNotNullorEmpty()] [int16]   $ReferencedThreshold = 2,
     [Parameter(Mandatory = $True,  Position = 3)]                           [String]  $ContentLocation
    )
  
    $CCMComObject                    = New-Object -ComObject 'UIResource.UIResourceMgr'
    $ThisFunctionName                = $MyInvocation.InvocationName
    $CacheInfo                       = @($CCMComObject.GetCacheInfo().GetCacheElements() | Where-Object { ($_.ContentID -eq $ContentID) -and ($_.Location -eq $ContentLocation) })
    $ReturnValue                     = $True
    $ReturnReason                    = ""
    [datetime]$OlderThan             = (Get-Date).ToUniversalTime().AddDays(-$ReferencedThreshold)
    $ReturnText                      = @{
    1 = $($CCMCacheActions.$Language.CCMDeleted_1)                                                           # The item has been removed successfully.
    2 = $($CCMCacheActions.$Language.CCMDeleted_2)                                                           # This item is used by other items, therefore not removed.
    3 = $($CCMCacheActions.$Language.CCMDeleted_3).Replace("<ReferencedThreshold>",$ReferencedThreshold)     # The content has been used in the last <ReferencedThreshold> days.
    4 = $($CCMCacheActions.$Language.CCMDeleted_4)                                                           # The item has not been removed successfully.
    5 = $($CCMCacheActions.$Language.CCMDeleted_5)                                                           # The item has no valid Last Reference Time, therefore skipped.
    }
    
    Add-EntryToLogFile -FunctionName $ThisFunctionName -Entry "----> Cache Element ID: $([string]$($CacheInfo.CacheElementID))"

  # =============================================================================================================================================
  # Only delete this entry if not referenced by other items and not used in the last number of <ReferencedThreshold> days.
  # =============================================================================================================================================

    If ($CacheInfo.LastReferenceTime)
     {
      If ( (($CacheInfo.ReferenceCount) -lt 1) -and ([datetime]($CacheInfo.LastReferenceTime) -le $OlderThan) )
       {
        $Result    = $CCMComObject.GetCacheInfo().DeleteCacheElementEx([string]$($CacheInfo.CacheElementID), [bool]$RemoveIfPersistentInCache)
        $CacheInfo = $CCMComObject.GetCacheInfo().GetCacheElements() | Where-Object { ($_.ContentID -eq $ContentID) -and ($_.Location -eq $ContentLocation) }

      # =============================================================================================================================================
      # Remove the object and check if the item has been deleted successfully.
      # The variable $Result is always empty. So you have the check by quering the CCM Object.
      # =============================================================================================================================================
      
        If ( -not($CacheInfo))    
          {
           $ReturnNumber = 1
           $ReturnValue  = $True
          } 
           else 
          {
           $ReturnNumber = 4
           $ReturnValue  = $False
          }
       }
        else
       {
        
      # =============================================================================================================================================
      # The content is used by other items or the content is used within the number of <ReferencedThreshold> days. 
      # So, checking. 
      # =============================================================================================================================================
        
        If (-not($CacheInfo.ReferenceCount) -lt 1)  
         {
          $ReturnNumber = 2
          $ReturnValue  = $False
         }
          else
         {
          $ReturnNumber = 3
          $ReturnValue  = $False
        } 
       }
     }
      else
     {
      
    # =============================================================================================================================================
    # No valid reference time.
    # =============================================================================================================================================
      
      $ReturnNumber = 5
      $ReturnValue  = $False 
     }

    Return $ReturnValue, $($ReturnText.Item($ReturnNumber))
   }

# =============================================================================================================================================
# End function block
# =============================================================================================================================================

# =============================================================================================================================================
# Functions, used in the forms blocks
# =============================================================================================================================================

  Function SelectOrUnselectAllItemsInDGV
   {

    <#
    .NOTES
    =============================================================================================================================================
    Created with:     Windows PowerShell ISE
    Functionname:     SelectOrUnselectAllItemsInDGV
    =============================================================================================================================================
    .SYNOPSIS

    Selects only the items that are in the local CCMCache. So these items can be cleared from the CCMCache.
        
    #>

    $ThisFunctionName                = $MyInvocation.InvocationName
    Add-EntryToLogFile -FunctionName $ThisFunctionName -Entry $($CCMCacheActions.$Language.PressedButtonName).Replace("<ButtonName>",$CCMCacheActions.$Language.chkAllItems)
    If ($chkAllItems.Checked)
     {
      $dgvCacheItems.SelectAll()
      ForEach ($Row in $dgvCacheItems.SelectedRows)
       {
        If ( -not($dgvCacheItems.Rows[$Row.Index].Cells['Content ID'].Value))
         {
          $dgvCacheItems.Rows[$Row.Index].Selected = $False
         }
       }
     }
      else
     {
      $dgvCacheItems.ClearSelection()
     }
   }

  Function FindAppsProgramsAndUpdates
   {
    
    <#
    .NOTES
    =============================================================================================================================================
    Created with:     Windows PowerShell ISE
    Functionname:     FindAppsProgramsAndUpdates
    =============================================================================================================================================
    .SYNOPSIS

    Calls the functions to find all the installed applications, programs and updates and fills the datagrid dgvCacheItems. 
        
    #>
    
    $ThisFunctionName = $MyInvocation.InvocationName
    $frmProcessing    = New-Object 'System.Windows.Forms.Form'
    $txtMessage       = New-Object 'System.Windows.Forms.TextBox'

    $frmProcessing.Controls.Add($txtMessage)
    $frmProcessing.AutoScaleDimensions     = New-Object System.Drawing.SizeF(6, 13)
    $frmProcessing.AutoScaleMode           = 'Font'
    $frmProcessing.BackColor               = [System.Drawing.SystemColors]::ControlDark 
    $frmProcessing.ClientSize              = New-Object System.Drawing.Size(430, 44)
    $frmProcessing.FormBorderStyle         = 'None'
    $frmProcessing.Name                    = 'frmProcessing'
    $frmProcessing.ShowIcon                = $False
    $frmProcessing.StartPosition           = 'CenterParent'
    $frmProcessing.Text = "Form"
    $frmProcessing.add_Shown({CCM-FindApplications
                              CCM-FindPrograms
                              CCM-FindUpdates            
                              [void]$frmProcessing.Close()
                              [void]$frmProcessing.Dispose()
                              })
    #
    # txtMessage
    #
    $txtMessage.ReadOnly                   = $True
    $txtMessage.Location                   = New-Object System.Drawing.Point(12, 12)
    $txtMessage.MaxLength                  = 50
    $txtMessage.Name                       = 'txtMessage'
    $txtMessage.Size                       = New-Object System.Drawing.Size(400, 20)
    $txtMessage.TabIndex                   = 0
    $txtMessage.Text                       = $CCMCacheActions.$Language.txtMessage
    $txtMessage.TextAlign                  = 'Center'
    $txtMessage.WordWrap                   = $False

    [void]$frmProcessing.ShowDialog()
        
    if ($Global:arrTable.Count -gt 0)
     {
      $Global:arrTable = $Global:arrTable | Sort-Object -Property "Name","Deployment Type","Program","Evaluation","Update Classification","Content ID","Content Location","Content Size","Content Last Reference Time" -Unique
      If ($chkOnlyCachedItems.Checked)
       {
        $tmpTable = $Global:arrTable | Where-Object {$_."Content ID"}
       }
        else
       {
        $tmpTable = $Global:arrTable
       }
      $dgvCacheItems.DataSource = $null
      $dgvCacheItems.Rows.Clear()
      $dgvCacheItems.DataSource = [System.Collections.ArrayList]$tmpTable
      $dgvCacheItems.SelectAll()
      ForEach ($Row in $dgvCacheItems.SelectedRows)
       {
        If (($dgvCacheItems.Rows[$Row.Index].Cells['Content ID'].Value).Length -eq 0)
         {
          $dgvCacheItems.Rows[$Row.Index].DefaultCellStyle.BackColor = [System.Drawing.Color]::FromArgb(255, 249, 249, 249)
          $dgvCacheItems.Rows[$Row.Index].DefaultCellStyle.ForeColor = [System.Drawing.Color]::FromArgb(255, 160, 160, 160)
         }
       }
      $dgvCacheItems.ClearSelection()
     }
   }

   Function ShowOnlyCachedItems
    {

     <#
     .NOTES
     =============================================================================================================================================
     Created with:     Windows PowerShell ISE
     Functionname:     ShowOnlyCachedItems
     =============================================================================================================================================
     .SYNOPSIS

     Displays only the items that are in the local CCMCache. So these items can be cleared from the CCMCache.
        
     #>

     $ThisFunctionName                = $MyInvocation.InvocationName
     Add-EntryToLogFile -FunctionName $ThisFunctionName -Entry $($CCMCacheActions.$Language.PressedButtonName).Replace("<ButtonName>",$CCMCacheActions.$Language.chkOnlyCachedItems).Replace("&","")

     $dgvCacheItems.DataSource = $null
     $dgvCacheItems.Rows.Clear()
     If ($chkOnlyCachedItems.Checked)
      {
       $tmpTable = $Global:arrTable | Where-Object {$_."Content ID"}
       Add-EntryToLogFile -FunctionName $ThisFunctionName -Entry $($CCMCacheActions.$Language.NumberOfItemsInCache).Replace("<Number>",$tmpTable.Count)
       if ($tmpTable.Count -gt 0)
        {
         $dgvCacheItems.DataSource = [System.Collections.ArrayList]$tmpTable
         $dgvCacheItems.Update()
        }
      }
       else
      {
       if ($Global:arrTable.Count -gt 0)
        {
         $dgvCacheItems.DataSource = [System.Collections.ArrayList]$Global:arrTable
         $dgvCacheItems.Update()
         $dgvCacheItems.SelectAll()
         ForEach ($Row in $dgvCacheItems.SelectedRows)
          {
           If (($dgvCacheItems.Rows[$Row.Index].Cells['Content ID'].Value).Length -eq 0)
            {
             $dgvCacheItems.Rows[$Row.Index].DefaultCellStyle.BackColor = [System.Drawing.Color]::FromArgb(255, 249, 249, 249)
             $dgvCacheItems.Rows[$Row.Index].DefaultCellStyle.ForeColor = [System.Drawing.Color]::FromArgb(255, 160, 160, 160)
            }
          }
         $dgvCacheItems.ClearSelection()
        }
      }
    }
  
  Function DeleteOrphanedFolders
   {

    <#
    .NOTES
    =============================================================================================================================================
    Created with:     Windows PowerShell ISE
    Functionname:     DeleteOrphanedFolders
    =============================================================================================================================================
    .SYNOPSIS

    Deletes the non used folders from the CCMCache.
        
    #>

    $ThisFunctionName                = $MyInvocation.InvocationName
    $ArrayOrphanedFolders            = @(CCM-FindOrphanedCCMFolders)
    Add-EntryToLogFile -FunctionName $ThisFunctionName -Entry $($CCMCacheActions.$Language.PressedButtonName).Replace("<ButtonName>",$CCMCacheActions.$Language.btnDeleteOrphanedFolders).Replace("&","")
    If ($ArrayOrphanedFolders.Count -gt 0)
     {
      $OrphanedFolders                 = $ArrayOrphanedFolders | ForEach-Object {$_."Orphaned Folder Name"}  | Out-String
      $Text                            = "$($CCMCacheActions.$Language.ConfirmRemovalFolders)`n`n$OrphanedFolders"
      $Result                          = Display-MessageBox -Text $Text -GUI -Button OkCancel
      If ($Result -eq 1)
       {
        ForEach ($OrphanedFolder in $ArrayOrphanedFolders)
         {
          if (-not $OrphanedFolder."ContentID")
           {
            Add-EntryToLogFile -Entry $($CCMCacheActions.$Language.DeletingFolder).Replace("<FolderName>",$($OrphanedFolder."Orphaned Folder Name")) -FunctionName $ThisFunctionName
            Remove-Item -Path $($OrphanedFolder."Orphaned Folder Name") -Recurse -Force
           }
            else
           {
            $Result,$ResultText = CCM-DeleteItemFromCache -ContentID $($OrphanedFolder."ContentID") -ContentLocation $($OrphanedFolder."Orphaned Folder Name") -RemoveIfPersistentInCache:$True -ReferencedThreshold 0
            Add-EntryToLogFile -FunctionName $ThisFunctionName -Entry "$($OrphanedFolder."Orphaned Folder Name") --> $ResultText"
           }
         }
        $btnDeleteOrphanedFolders.Enabled = $False
       }
     }
      else
     {
      $Text       = $CCMCacheActions.$Language.NoOrphanedFolders
      Add-EntryToLogFile -FunctionName $ThisFunctionName -Entry $Text 
      $Result     = Display-MessageBox -Text $Text -GUI
     }
   }

  Function RemoveItemsFromCache
   {

    <#
    .NOTES
    =============================================================================================================================================
    Created with:     Windows PowerShell ISE
    Functionname:     RemoveItemsFromCache
    =============================================================================================================================================
    .SYNOPSIS

    Removes an item from the CCMCache.
        
    #>

    $ThisFunctionName                = $MyInvocation.InvocationName
    Add-EntryToLogFile -FunctionName $ThisFunctionName -Entry $($CCMCacheActions.$Language.PressedButtonName).Replace("<ButtonName>",$CCMCacheActions.$Language.btnClearCache).Replace("&","")
    $Text = "$($CCMCacheActions.$Language.RemovedFromCache)`n"
    $Rerun_FindAppsProgramsAndUpdates = $True
    ForEach($Row in $dgvCacheItems.SelectedRows)
     {
      $SelectedContentID       = $dgvCacheItems.Rows[$Row.Index].Cells['Content ID'].Value
      $SelectedContentLocation = $dgvCacheItems.Rows[$Row.Index].Cells['Content Location'].Value
      If ($SelectedContentID)
       {
        $Result,$ResultText = CCM-DeleteItemFromCache -ContentID $SelectedContentID -ContentLocation $SelectedContentLocation
        If ($Result)
         {
          $Line = $($CCMCacheActions.$Language.RemovedFromCacheSuccess).Replace("<Cell>",$($dgvCacheItems.Rows[$Row.Index].Cells['Name'].Value))
          Add-EntryToLogFile -FunctionName $ThisFunctionName -Entry $Line
         }
          else
         {
          $Line = $($CCMCacheActions.$Language.RemovedFromCacheFailure).Replace("<Cell>",$($dgvCacheItems.Rows[$Row.Index].Cells['Name'].Value)).Replace("<ResultText>",$ResultText)
          Add-EntryToLogFile -FunctionName $ThisFunctionName -Entry $Line
          $Rerun_FindAppsProgramsAndUpdates = $False
         }
        }
         else
        {
         $Line = $($CCMCacheActions.$Language.ContentIDNotFoundInCache).Replace("<Cell>",$($dgvCacheItems.Rows[$Row.Index].Cells['Name'].Value))
         Add-EntryToLogFile -FunctionName $ThisFunctionName -Entry $Line
         $Rerun_FindAppsProgramsAndUpdates = $False
        }
      $Text +="`n$Line`n"
     }
    $Result = Display-MessageBox -Text $Text -GUI -Button Ok
    If ($Rerun_FindAppsProgramsAndUpdates)
     {
      $Global:arrTable = @()
      FindAppsProgramsAndUpdates
     }
   }

# =============================================================================================================================================
# End functions, used in the forms blocks.
# =============================================================================================================================================

  # Clear-Host

  $Global:gblDetailedLogging                     = $DetailedLogging
  $strCurrentDir                                 = Split-Path -parent $MyInvocation.MyCommand.Definition
  $ApplicationName                               = "Clear Software Center Cache"
  $ApplicationVersion                            = "0.1 (Beta 2)"
  $PowerShellLanguageMode                        = $ExecutionContext.SessionState.LanguageMode
  $Global:bolFullLanguage                        = $PowerShellLanguageMode -eq "FullLanguage"
  $Global:bolElevatedRights                      = Check-HasElevatedRights
  $MinimumWidth                                  = 1000
  $MinimumHeight                                 = 700
  $ErrorNumber                                   = 0
  $ErrorText                                     = ""
  $Global:arrTable                               = @()

# =============================================================================================================================================
# Find the logpath.
# It is the key 'HKEY_CURRENT_USER\SOFTWARE\Microsoft\Windows\CurrentVersion\Explorer\Shell Folders' with the value 'Local AppData'.
# And then '\temp' is added. 
# =============================================================================================================================================

  $OnlyUserName,               `
  $LoggedOnUserInDomainFormat, `
  $UseridSID,                  `
  $InstallAccount                 = UserDetails
  $RegKey                         = "REGISTRY::HKEY_USERS\$UseridSID\SOFTWARE\Microsoft\Windows\CurrentVersion\Explorer\Shell Folders"
  $ValueName                      = "Local AppData"
  $Value                          = (Get-ItemProperty $RegKey).$ValueName
  $LogPath                        = $Value + "\temp"    
  $Global:gblLogPath              = $LogPath

# =============================================================================================================================================
# Define the results file. This file contains all the results.
# =============================================================================================================================================
 
  $strLastPartOfFileName                         = " ($((Get-Date).ToString('G')))"
  $strLastPartOfFileName                         = $strLastPartOfFileName -replace ":","-"
  $strLastPartOfFileName                         = $strLastPartOfFileName -replace "/","-"
  $PreFixLogFile                                 = $ApplicationName -replace(" ","")
  $ThisFunctionName                              = "[Main - Resultsfile]"
   
  If ($Global:gblDetailedLogging)
   {
    $Global:gblLogFile = $Global:gblLogPath + "\"+ $PreFixLogFile + $($strLastPartOfFileName + ".log")
    Add-EntryToLogFile -FunctionName $ThisFunctionName -Entry "Logfile: $Global:gblLogFile"
   }

# =============================================================================================================================================
# Find all the arguments and put them in the log file
# Source: https://ss64.com/ps/psboundparameters.html
# =============================================================================================================================================

  $ThisFunctionName                              = "[Main - Arguments]"
  If ($Global:bolFullLanguage)
   {
    $TableWithParameters  = @()
    $RecordWithParameters = [ordered] @{"Key"     = "";
                                        "Value"   = ""}
  
    Add-EntryToLogFile -FunctionName $ThisFunctionName -Entry "***** Parameters part           *****"
  
    ForEach($boundparam in $PSBoundParameters.GetEnumerator()) 
     {
      $tmpValue                     = $($boundparam.Value)
      $Value                        = ""
      If ($tmpValue -is [array])
       {
        ForEach ($object in $tmpValue)
         {
          If (-not($value))
           {
            $Value = $object
           }
            else
           {
            $Value +=",$($object)"
           }
         }
       }
        else
       {
        $Value = $tmpValue
       }
      Add-EntryToLogFile -FunctionName $ThisFunctionName -Entry "Key: $($boundparam.Key) Value: $Value" 
     }
    Add-EntryToLogFile -FunctionName $ThisFunctionName -Entry "***** End Parameters part       *****`r`n" 
   }
    else
   {
    Add-EntryToLogFile -FunctionName $ThisFunctionName -Entry "The parameters are not written to the logfile due to PowerShell ConstrainedMode."
   }

# =============================================================================================================================================
# Write the logged in user details to the log file. 
# =============================================================================================================================================
  
  $ThisFunctionName                              = "[Main - User Details]"
  Add-EntryToLogFile -FunctionName $ThisFunctionName -Entry "***** User details part         *****" 
  Add-EntryToLogFile -FunctionName $ThisFunctionName -Entry "Logged on user:                  $LoggedOnUserInDomainFormat"
  Add-EntryToLogFile -FunctionName $ThisFunctionName -Entry "Logged on user (SID):            $UseridSID"
  Add-EntryToLogFile -FunctionName $ThisFunctionName -Entry "Has (elevated) admin rights:     $Global:bolElevatedRights"
  Add-EntryToLogFile -FunctionName $ThisFunctionName -Entry "Installation account:            $InstallAccount"
  If ($Global:gblDetailedLogging)
   {
    Add-EntryToLogFile -FunctionName $ThisFunctionName -Entry "Logfile:                         $Global:gblLogFile"
   }
  Add-EntryToLogFile -FunctionName $ThisFunctionName -Entry "***** End User details part     *****`r`n"

# =============================================================================================================================================
# Read the JSON file with the translations.
# If the JSON file does not contain the detected language, then fallback to English.
# =============================================================================================================================================

  $ThisFunctionName                              = "[Main - Language]"
  Add-EntryToLogFile -FunctionName $ThisFunctionName -Entry "***** Language part             *****"

  if ($LanguageOverride)
   {
    $Language = $LanguageOverride
    Add-EntryToLogFile -FunctionName $ThisFunctionName  -Entry "The parameter -LanguageOverride is used. The language is '$Language'."
   }
    else
   {
    $Language = (Find-Language -CurrentUserSID $UseridSID).SubString(0, 2)
   }

  $JSONFile = $strCurrentDir + "\"+ $($MyInvocation.MyCommand.Name -replace ".ps1",".json")

  if ( -not(Test-Path($JSONFile)))
   {
    $Message = "The language file '$JSONFile' does not exists. Leaving the script."
    Add-EntryToLogFile -FunctionName $ThisFunctionName -Entry $Message
    Write-Host $Message -BackgroundColor Red -ForegroundColor White
    Exit 1
   }

  $CCMCacheActions = Get-Content $JSONFile -Encoding UTF8 | ConvertFrom-Json

  if ( -not($CCMCacheActions.$Language))
   {
    Add-EntryToLogFile -FunctionName $ThisFunctionName -Entry "The language '$Language' is not found in the json file '$JSONFile'."
    $Language = "en"
    Add-EntryToLogFile -FunctionName $ThisFunctionName -Entry "Falling back to the default language '$Language'."
   }

  Add-EntryToLogFile -FunctionName $ThisFunctionName -Entry "The language '$Language' is used."

  Add-EntryToLogFile -FunctionName $ThisFunctionName -Entry "***** End language part         *****`r`n" 

# =============================================================================================================================================
# End reading the JSON file with the translations.
# =============================================================================================================================================

# =============================================================================================================================================
# Check if there is a (working) SCCM Client installed.
# =============================================================================================================================================

  $BoleanSCCMClient,$ErrorMessageRegardingSCCMClient = Check-SCCMClient

# =============================================================================================================================================
# End checking if there is a (working) SCCM Client installed.
# =============================================================================================================================================

# =============================================================================================================================================
# Add assembly (only FullLanguage)
# =============================================================================================================================================
  
  If ($Global:bolFullLanguage)
   {    
    Add-Type -AssemblyName System.Windows.Forms
    Add-Type -AssemblyName System.Drawing
    Add-Type -AssemblyName System.Runtime
    Add-Type -AssemblyName PresentationFramework 
    [System.Windows.Forms.Application]::EnableVisualStyles()
    $DetectedWidth  = ([System.Windows.Forms.SystemInformation]::PrimaryMonitorSize).Width
    $DetectedHeight = ([System.Windows.Forms.SystemInformation]::PrimaryMonitorSize).Height
    If ($DetectedWidth -le $MinimumWidth)   {$ErrorNumber = $ErrorNumber + 16}
    If ($DetectedHeight -le $MinimumHeight) {$ErrorNumber = $ErrorNumber + 32}
   }

# =============================================================================================================================================
# Error handling.
# =============================================================================================================================================

  If (-not $Global:bolFullLanguage)   {$ErrorNumber = $ErrorNumber +  2}
  if (-not $BoleanSCCMClient)         {$ErrorNumber = $ErrorNumber +  4}
  if (-not $Global:bolElevatedRights) {$ErrorNumber = $ErrorNumber +  8} 
  if (($ErrorNumber -band 2) -eq 2)   {$ErrorText = $CCMCacheActions.$Language.Error02Text}
  if (($ErrorNumber -band 4) -eq 4)
   {
    $tmpText = $($CCMCacheActions.$Language.Error04Text).Replace("<ErrorSCCMClient>",$ErrorMessageRegardingSCCMClient).Trim()
    If ($ErrorText) {$ErrorText += "`n$tmpText"} else {$ErrorText = $tmpText}
   }
  
  if (($ErrorNumber -band 8) -eq 8)
   {
    If ($ErrorText) {$ErrorText += "`n$($CCMCacheActions.$Language.Error08Text)"} else {$ErrorText = $($CCMCacheActions.$Language.Error08Text)}
   }

  if (($ErrorNumber -band 16) -eq 16)
   {
    $tmpText = $($CCMCacheActions.$Language.Error16Text).Replace("<DetectedWidth>",$DetectedWidth).Replace("<AllowedWidth>",$MinimumWidth)
    If ($ErrorText) {$ErrorText += "`n$tmpText"} else {$ErrorText = $tmpText}
   }
  
  if (($ErrorNumber -band 32) -eq 32)
   {
    $tmpText = $($CCMCacheActions.$Language.Error32Text).Replace("<DetectedHeight>",$DetectedHeight).Replace("<AllowedHeight>",$MinimumHeight)
    If ($ErrorText) {$ErrorText += "`n$tmpText"} else {$ErrorText = $tmpText}
   }

  If ($ErrorNumber -gt 0)
   {
    $DummyValue = Display-MessageBox -Text $ErrorText -GUI:$Global:bolFullLanguage -Error
    Exit $ErrorNumber
   }

# =============================================================================================================================================
# End Error handling.
# =============================================================================================================================================

  $frmClearSCCache                                           = New-Object 'System.Windows.Forms.Form'
  $btnDeleteOrphanedFolders                                  = New-Object 'System.Windows.Forms.Button'
  $chkAllItems                                               = New-Object 'System.Windows.Forms.CheckBox'
  $btnCancel                                                 = New-Object 'System.Windows.Forms.Button'
  $btnClearCache                                             = New-Object 'System.Windows.Forms.Button'
  $chkOnlyCachedItems                                        = New-Object 'System.Windows.Forms.CheckBox'
  $dgvCacheItems                                             = New-Object 'System.Windows.Forms.DataGridView'
  $lblDiscoveredItems                                        = New-Object 'System.Windows.Forms.Label'

  #
  # frmClearSCCache
  #
  $frmClearSCCache.Controls.Add($btnDeleteOrphanedFolders)
  $frmClearSCCache.Controls.Add($chkAllItems)
  $frmClearSCCache.Controls.Add($btnCancel)
  $frmClearSCCache.Controls.Add($btnClearCache)
  $frmClearSCCache.Controls.Add($chkOnlyCachedItems)
  $frmClearSCCache.Controls.Add($dgvCacheItems)
  $frmClearSCCache.Controls.Add($lblDiscoveredItems)
  $frmClearSCCache.AutoScaleDimensions                       = New-Object System.Drawing.SizeF(6, 13)
  $frmClearSCCache.AutoScaleMode                             = 'Font'
  $frmClearSCCache.ClientSize                                = New-Object System.Drawing.Size(984, 661)
  $frmClearSCCache.FormBorderStyle                           = 'FixedSingle'
  $frmClearSCCache.Name                                      = 'frmClearSCCache'
  $frmClearSCCache.StartPosition                             = 'CenterScreen'
  $frmClearSCCache.Text                                      = "$($CCMCacheActions.$Language.frmClearSCCache) $ApplicationVersion"
  $frmClearSCCache.add_Load($frmClearSCCache_Load)
  $frmClearSCCache.MaximizeBox                               = $False
  $frmClearSCCache.MinimizeBox                               = $False
  # 
  # btnDeleteOrphanedFolders
  #
  $btnDeleteOrphanedFolders.Location                         = New-Object System.Drawing.Point(540, 604)
  $btnDeleteOrphanedFolders.Name                             = 'btnDeleteOrphanedFolders'
  $btnDeleteOrphanedFolders.Size                             = New-Object System.Drawing.Size(140, 45)
  $btnDeleteOrphanedFolders.TabIndex                         = 6
  $btnDeleteOrphanedFolders.Text                             = $CCMCacheActions.$Language.btnDeleteOrphanedFolders
  $btnDeleteOrphanedFolders.UseVisualStyleBackColor          = $True
  #
  # chkAllItems
  #
  $chkAllItems.Location                                      = New-Object System.Drawing.Point(12, 625)
  $chkAllItems.Name                                          = 'chkAllItems'
  $chkAllItems.Size                                          = New-Object System.Drawing.Size(253, 24)
  $chkAllItems.TabIndex                                      = 5
  $chkAllItems.Text                                          = $CCMCacheActions.$Language.chkAllItems
  $chkAllItems.UseVisualStyleBackColor                       = $True
  #
  # btnCancel
  #
  $btnCancel.Location                                        = New-Object System.Drawing.Point(832, 604)
  $btnCancel.Name                                            = 'btnCancel'
  $btnCancel.Size                                            = New-Object System.Drawing.Size(140, 45)
  $btnCancel.TabIndex                                        = 4
  $btnCancel.Text                                            = $CCMCacheActions.$Language.btnCancel
  $btnCancel.UseVisualStyleBackColor                         = $True
  #
  # btnClearCache
  #
  $btnClearCache.Location                                    = New-Object System.Drawing.Point(686, 604)
  $btnClearCache.Name                                        = 'btnClearCache'
  $btnClearCache.Size                                        = New-Object System.Drawing.Size(140, 45)
  $btnClearCache.TabIndex                                    = 3
  $btnClearCache.Text                                        = $CCMCacheActions.$Language.btnClearCache
  $btnClearCache.UseVisualStyleBackColor                     = $True
  #
  # chkOnlyCachedItems
  #
  $chkOnlyCachedItems.Location                               = New-Object System.Drawing.Point(13, 604)
  $chkOnlyCachedItems.Name                                   = 'chkOnlyCachedItems'
  $chkOnlyCachedItems.Size                                   = New-Object System.Drawing.Size(341, 24)
  $chkOnlyCachedItems.TabIndex                               = 2
  $chkOnlyCachedItems.Text                                   = $CCMCacheActions.$Language.chkOnlyCachedItems
  $chkOnlyCachedItems.UseVisualStyleBackColor                = $True
  #
  # dgvCacheItems
  #
  $dgvCacheItems.AllowUserToAddRows                          = $False
  $dgvCacheItems.AllowUserToDeleteRows                       = $False
  $dgvCacheItems.AllowUserToResizeRows                       = $False
  $dgvCacheItems.AutoSizeColumnsMode                         = 'AllCells'
  $dgvCacheItems.AutoSizeRowsMode                            = 'AllCells'
  $dgvCacheItems.ClipboardCopyMode                           = 'Disable'
  $dgvCacheItems.ColumnHeadersHeightSizeMode                 = 'AutoSize'
  $dgvCacheItems.Location                                    = New-Object System.Drawing.Point(13, 36)
  $dgvCacheItems.Name                                        = 'dgvCacheItems'
  $dgvCacheItems.RowTemplate.Resizable                       = 'True'
  $dgvCacheItems.SelectionMode                               = 'FullRowSelect'
  $dgvCacheItems.ShowEditingIcon                             = $False
  $dgvCacheItems.Size                                        = New-Object System.Drawing.Size(959, 562)
  $dgvCacheItems.TabIndex                                    = 1
  $dgvCacheItems.ReadOnly                                    = $True
  #
  # lblDiscoveredItems
  #
  $lblDiscoveredItems.AutoSize                               = $True
  $lblDiscoveredItems.Location                               = New-Object System.Drawing.Point(12, 9)
  $lblDiscoveredItems.Name                                   = 'lblDiscoveredItems'
  $lblDiscoveredItems.Size                                   = New-Object System.Drawing.Size(235, 13)
  $lblDiscoveredItems.TabIndex                               = 0
  $lblDiscoveredItems.Text                                   = $CCMCacheActions.$Language.lblDiscoveredItems
  
# =============================================================================================================================================
# All kind of actions
# =============================================================================================================================================

  $btnDeleteOrphanedFolders.Enabled                          = $False
  $chkOnlyCachedItems.Enabled                                = $False
  $btnClearCache.Enabled                                     = $False

  $frmClearSCCache.add_Shown({FindAppsProgramsAndUpdates
                              If ((CCM-FindOrphanedCCMFolders | Measure-Object).Count -gt 0)         {$btnDeleteOrphanedFolders.Enabled = $True}
                              If ($($Global:arrTable | Where-Object {$_."Content ID"}).Count -gt 0)  {$chkOnlyCachedItems.Enabled       = $True; $btnClearCache.Enabled = $True}
                              })


  $chkAllItems.Add_CheckedChanged({SelectOrUnselectAllItemsInDGV})
  $chkOnlyCachedItems.Add_CheckedChanged({ShowOnlyCachedItems})

  $dgvCacheItems.Add_SelectionChanged({
  
# =============================================================================================================================================
# If there is no Content ID then the whole row should be greyed out. 
# See https://stackoverflow.com/questions/72535521/powershell-datagridview-avoid-selection-of-rows-based-on-a-value-in-cell 
# for more background information. 
# =============================================================================================================================================
   
   ForEach ($Row in $dgvCacheItems.SelectedRows)
    {     
     If (($Row.Cells['Content ID'].Value).Length -eq 0)
      {
       $Row.Selected = $False     
       $dgvCacheItems.Rows[$Row.Index].DefaultCellStyle.BackColor = [System.Drawing.Color]::FromArgb(255, 249, 249, 249)
       $dgvCacheItems.Rows[$Row.Index].DefaultCellStyle.ForeColor = [System.Drawing.Color]::FromArgb(255, 160, 160, 160)
      }
      else
      {
       $dgvCacheItems.Rows[$Row.Index].DefaultCellStyle.BackColor = [System.Drawing.Color]::FromArgb(255, 255, 255, 255)
       $dgvCacheItems.Rows[$Row.Index].DefaultCellStyle.ForeColor = [System.Drawing.Color]::FromArgb(255, 0  , 0  , 0)
      }
    }
  })

  $btnCancel.add_Click({
   Add-EntryToLogFile -FunctionName $ThisFunctionName -Entry $($CCMCacheActions.$Language.PressedButtonName).Replace("<ButtonName>",$CCMCacheActions.$Language.btnCancel).Replace("&","")
   [void]$frmClearSCCache.Close()
   [void]$frmClearSCCache.Dispose()})

  $btnDeleteOrphanedFolders.add_Click({DeleteOrphanedFolders})

  $btnClearCache.add_Click({RemoveItemsFromCache})
  
  [void]$frmClearSCCache.ShowDialog()

# =============================================================================================================================================
# End all kind of actions.
# =============================================================================================================================================
